<?php
/*
Plugin Name: HF WP Scan 
Description: Escanea archivos PHP y lista los que usan funciones potencialmente peligrosas, 'eval','exec','system','base64_decode' y exportación a CSV.
Version: 1.2
Author: Ariel Hernández Friz
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

/*
WP Scan Security Mejorado es un software libre; puedes redistribuirlo y/o modificarlo bajo los términos 
de la Licencia Pública General de GNU según sea publicada por la Free Software Foundation; 
ya sea la versión 2 de la Licencia, o (a tu elección) cualquier versión posterior.

WP Scan Security Mejorado se distribuye con la esperanza de que sea útil, pero SIN NINGUNA GARANTÍA; 
sin siquiera la garantía implícita de COMERCIALIZACIÓN o IDONEIDAD PARA UN PROPÓSITO PARTICULAR. 
Consulta la Licencia Pública General de GNU para más detalles.

Debes haber recibido una copia de la Licencia Pública General de GNU junto con este plugin; 
si no, visita https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) exit;

// Menú en el dashboard
add_action('admin_menu', 'wpss_add_admin_page');
function wpss_add_admin_page() {
    add_menu_page(
        'WP Scan Security',
        'WP Scan Security',
        'manage_options',
        'wp-scan-security',
        'wpss_admin_page',
        'dashicons-shield-alt',
        100
    );
}

// Encolar JS
add_action('admin_enqueue_scripts', 'wpss_enqueue_scripts');

function wpss_enqueue_scripts($hook){
    if ($hook === 'toplevel_page_wp-scan-security') {
        wp_enqueue_script(
            'wpss_js',
            plugin_dir_url(__FILE__) . 'assets/wpss.js',
            ['jquery'],
            '1.0',
            true
        );
        wp_localize_script('wpss_js', 'wpss_ajax', [
            'ajax_url' => admin_url('admin-ajax.php')
        ]);
    }
}

// Página del plugin
function wpss_admin_page() {
    if (!current_user_can('manage_options')) return;

    echo '<div class="wrap"><h1>WP Scan Security Mejorado</h1>';
    echo '<p>Selecciona la carpeta que deseas escanear:</p>';
    
    echo '<form id="wpss_form">';
    $options = [
        'wp-admin' => 'wp-admin',
        'wp-content' => 'wp-content',
        'wp-includes' => 'wp-includes',
        'all' => 'Todo el sitio'
    ];
    foreach ($options as $key => $label) {
        echo '<label><input type="radio" name="folder" value="'.esc_attr($key).'"> '.$label.'</label><br>';
    }


    // --- Exclusión de plugins ---
    echo '<div id="wpss_plugins_exclude" style="margin-top:15px; display:none;">';
    echo '<p><strong>Excluir plugins del escaneo:</strong></p>';
    
    if ( function_exists('get_plugins') ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        $plugins = get_plugins();
        foreach ($plugins as $path => $plugin) {
            $slug = dirname($path);
            echo '<label style="display:block;margin-left:15px;">';
            echo '<input type="checkbox" name="exclude_plugins[]" value="'.esc_attr($slug).'"> ';
            echo esc_html($plugin['Name']);
            echo '</label>';
        }
    } else {
        echo '<p>No se pudieron cargar los plugins.</p>';
    }
    echo '</div>';


    echo '<br><input type="button" id="wpss_scan" class="button button-primary" value="Escanear">';
    echo '</form><hr>';

    echo '<div id="wpss_results"></div>';
    echo '<button id="wpss_export" class="button button-secondary" style="display:none;">Exportar CSV</button>';
    echo '</div>';
}




// Función de escaneo (AJAX) optimizada
add_action('wp_ajax_wpss_scan', 'wpss_ajax_scan');
function wpss_ajax_scan() {
    if (!current_user_can('manage_options')) wp_die('No permitido');

    $folder = isset($_POST['folder']) ? sanitize_text_field($_POST['folder']) : 'all';
    $dir = ABSPATH;

    if (in_array($folder, ['wp-admin','wp-content','wp-includes'])) {
        $dir .= $folder . '/';
    }

    // Plugins a excluir (solo se aplicará en wp-content/plugins)
    $excluded_plugins = isset($_POST['exclude_plugins']) ? array_map('sanitize_text_field', $_POST['exclude_plugins']) : [];
    $plugin_dir = plugin_dir_path(__FILE__);

    $flags = ['eval','exec','system','base64_decode'];
    $advanced_patterns = [
        '/eval\s*\(\s*base64_decode\s*\(/i',           // eval(base64_decode(...))
        '/preg_replace\s*\(.+,\s*.+,\s*.+,\s*e\s*\)/i', // preg_replace con /e
        '/include\s*\(\s*\$_(GET|POST|REQUEST)\s*\[.*\]\s*\)/i' // inclusión remota
    ];

    $files_found = [];

    $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));

    foreach ($iterator as $file) {
        if (!$file->isFile() || pathinfo($file, PATHINFO_EXTENSION) !== 'php') continue;

        $filepath = $file->getRealPath();

        // Excluir el propio plugin siempre
        if (strpos($filepath, $plugin_dir) === 0) continue;

        // Solo si estamos dentro de wp-content/plugins aplicamos exclusión de plugins
        if ($folder === 'wp-content' || $folder === 'all') {
            $rel_path = str_replace(ABSPATH,'',$filepath);
            if (preg_match('#^wp-content/plugins/([^/]+)#', $rel_path, $matches)) {
                if (in_array($matches[1], $excluded_plugins)) continue;
            }
        }

        $content = file_get_contents($filepath);
        $matches = [];

        // Flags simples
        foreach ($flags as $flag) {
            if (strpos($content, $flag) !== false) $matches[] = $flag;
        }

        // Patrones avanzados
        foreach ($advanced_patterns as $pattern) {
            if (preg_match($pattern, $content)) $matches[] = "Patrón avanzado detectado";
        }

        if ($matches) {
            $files_found[] = [
                'file' => str_replace(ABSPATH,'',$filepath),
                'flags' => $matches,
                'modified' => date("Y-m-d H:i:s", filemtime($filepath))
            ];
        }
    }

    // Ordenar por fecha más reciente
    usort($files_found, function($a, $b) {
        return strtotime($b['modified']) - strtotime($a['modified']);
    });

    wp_send_json($files_found);
}
?>